# Copyright (c) 2025, benilerouge.ddns.net
# Licensed under the MIT License.

import tkinter as tk
from tkinter import filedialog, messagebox, simpledialog, ttk
import tkinter.font as tkFont
import os
import re
import tempfile
import threading
import subprocess
import sys  # Pour obtenir l'interpréteur Python utilisé


class VideoGpxApp:
    def __init__(self, parent, get_video_path=None):
        self.parent = parent
        self.get_video_path = get_video_path or (lambda: None)
        self.build_interface()

    def build_interface(self):
        frm = tk.Frame(self.parent, padx=4, pady=4)
        frm.pack(fill="both", expand=True)

        btns = tk.Frame(frm)
        btns.pack(anchor="w", pady=6)

        font_bouton = tkFont.Font(family="Helvetica", size=14, weight="bold")

        btn_concat = tk.Button(
            btns,
            text="Concaténer des vidéos 🎞️",
            command=self.run_concat_videos,
            width=24,
            height=2,
            font=font_bouton
        )
        btn_concat.pack(side="left", padx=(0, 10))

        btn_gpx = tk.Button(
            btns,
            text="Lisser un fichier FIT/GPX 📈️",
            command=self.run_fit_gpx_processing,
            width=24,
            height=2,
            font=font_bouton
        )
        btn_gpx.pack(side="left", padx=(0, 10))

        btn_editeurgpx = tk.Button(
            btns,
            text="Éditeur GPX️",
            command=self.run_editeurgpx,
            width=24,
            height=2,
            font=font_bouton
        )
        btn_editeurgpx.pack(side="left", padx=(0, 10))

    def run_fit_gpx_processing(self):
        filepath = filedialog.askopenfilename(
            title="Ouvrir un fichier FIT ou GPX",
            filetypes=[("Fichiers FIT/GPX", "*.fit *.FIT *.gpx *.GPX")],
            initialdir=os.getcwd()
        )
        if not filepath or not os.path.isfile(filepath):
            return
        ext = os.path.splitext(filepath)[1].lower()

        window = simpledialog.askinteger("Lissage altitude", "Taille de la fenêtre (impair, ex: 11):", initialvalue=11)
        if not window or window < 1 or window % 2 == 0:
            window = 11
        seuil = simpledialog.askfloat("Seuil altitude", "Seuil saut invraisemblable (mètres):", initialvalue=10.0)
        if seuil is None or seuil <= 0:
            seuil = 10.0

        saveas = filedialog.asksaveasfilename(
            title="Fichier GPX lissé",
            defaultextension=".gpx",
            initialfile=os.path.splitext(os.path.basename(filepath))[0] + "_smooth.gpx",
            filetypes=[("GPX", "*.gpx")],
            initialdir=os.getcwd()
        )
        if not saveas:
            return

        try:
            if ext == ".gpx":
                # Traitement direct GPX
                out = self.lisse_gpx_file(filepath, saveas, window, seuil)
            elif ext == ".fit":
                # Conversion FIT -> GPX via gpsbabel
                tmp_gpx = self.convert_fit_to_gpx_with_gpsbabel(filepath)
                out = self.lisse_gpx_file(tmp_gpx, saveas, window, seuil)
                os.remove(tmp_gpx)
            else:
                messagebox.showerror("Erreur", "Format de fichier non supporté.")
                return
            messagebox.showinfo("Lissage terminé", f"Lissage terminé !\nFichier généré :\n{out}")
        except Exception as e:
            messagebox.showerror("Erreur", str(e))

    def convert_fit_to_gpx_with_gpsbabel(self, fit_path):
        """
        Convertit un fichier FIT en GPX en utilisant gpsbabel.
        """
        tmp_gpx = tempfile.NamedTemporaryFile(delete=False, suffix=".gpx")
        tmp_gpx.close()

        cmd = [
            "gpsbabel",
            "-i", "garmin_fit",
            "-f", fit_path,
            "-x", "duplicate",
            "-o", "gpx",
            "-F", tmp_gpx.name
        ]

        try:
            subprocess.run(cmd, check=True, capture_output=True)
        except subprocess.CalledProcessError as e:
            err_msg = e.stderr.decode() if e.stderr else str(e)
            raise RuntimeError(f"Erreur lors de la conversion FIT→GPX avec gpsbabel :\n{err_msg}")

        return tmp_gpx.name

    @staticmethod
    def extract_ele_list(gpx_content):
        return [float(val) for val in re.findall(r"<ele>(-?[0-9.]+)</ele>", gpx_content)]

    @staticmethod
    def clean_outliers(ele_list, threshold):
        cleaned = ele_list.copy()
        n = len(ele_list)
        for i in range(1, n-1):
            prev = ele_list[i-1]
            curr = ele_list[i]
            next_ = ele_list[i+1]
            if abs(curr - prev) > threshold and abs(curr - next_) > threshold:
                cleaned[i] = (prev + next_) / 2
        return cleaned

    @staticmethod
    def moving_average(data, w):
        half = w // 2
        res = []
        for i in range(len(data)):
            start = max(0, i - half)
            end = min(len(data), i + half + 1)
            res.append(sum(data[start:end]) / (end - start))
        return res

    @staticmethod
    def lisse_gpx_file(input_file, output_file, window=5, seuil=10):
        with open(input_file, "r", encoding="utf-8") as f:
            gpx = f.read()
        eles = VideoGpxApp.extract_ele_list(gpx)
        eles_clean = VideoGpxApp.clean_outliers(eles, seuil)
        eles_smooth = VideoGpxApp.moving_average(eles_clean, window)

        def replace_ele(match):
            idx = replace_ele.idx
            val = eles_smooth[idx]
            replace_ele.idx += 1
            return f"<ele>{val:.2f}</ele>"

        replace_ele.idx = 0
        gpx_smooth = re.sub(r"<ele>-?[0-9.]+</ele>", replace_ele, gpx)
        with open(output_file, "w", encoding="utf-8") as f:
            f.write(gpx_smooth)
        return output_file

    def run_editeurgpx(self):
        script_path = os.path.join(os.path.dirname(os.path.abspath(__file__)), "GPXcutter.py")
        script_dir = os.path.dirname(script_path)

        if not os.path.isfile(script_path):
            messagebox.showerror("Erreur", "Le fichier GPXcutter.py est introuvable.")
            return

        def run_script():
            try:
                subprocess.run([sys.executable, script_path], check=True, cwd=script_dir)
                self.parent.after(0, lambda: messagebox.showinfo("Succès", "GPXcutter.py a été exécuté avec succès."))
            except subprocess.CalledProcessError as e:
                self.parent.after(0, lambda e=e: messagebox.showerror("Erreur", f"Erreur d'exécution de GPXcutter :\n{e}"))
            except FileNotFoundError as e:
                self.parent.after(0, lambda: messagebox.showerror("Erreur", f"Interpréteur Python introuvable :\n{e}"))

        threading.Thread(target=run_script, daemon=True).start()

    def run_concat_videos(self):
        self.concat_window = tk.Toplevel(self.parent)
        self.concat_window.title("Concaténation - Gestion des fichiers")
        self.concat_window.geometry("600x400")
        self.concat_window.transient(self.parent)
        self.concat_window.grab_set()

        frm = tk.Frame(self.concat_window, padx=10, pady=10)
        frm.pack(fill="both", expand=True)

        self.listbox_files = tk.Listbox(frm, selectmode=tk.EXTENDED, font=("Arial", 11))
        scrollbar = tk.Scrollbar(frm, orient="vertical", command=self.listbox_files.yview)
        self.listbox_files.config(yscrollcommand=scrollbar.set)
        self.listbox_files.pack(side="left", fill="both", expand=True)
        scrollbar.pack(side="left", fill="y")

        btns_frame = tk.Frame(frm, padx=10)
        btns_frame.pack(side="left", fill="y", anchor="n")

        font_bouton = tkFont.Font(family="Helvetica", size=14, weight="bold")

        btn_add = tk.Button(btns_frame, text="Ajouter fichiers", width=18, font=font_bouton, command=self.concat_add_files)
        btn_add.pack(pady=(0, 10))

        btn_remove = tk.Button(btns_frame, text="Supprimer sélection", width=18, font=font_bouton, command=self.concat_remove_selected)
        btn_remove.pack(pady=(0, 10))

        btn_up = tk.Button(btns_frame, text="Monter", width=18, font=font_bouton, command=self.concat_move_up)
        btn_up.pack(pady=(0, 10))

        btn_down = tk.Button(btns_frame, text="Descendre", width=18, font=font_bouton, command=self.concat_move_down)
        btn_down.pack(pady=(0, 10))

        bottom_frame = tk.Frame(self.concat_window, pady=10)
        bottom_frame.pack(fill="x", side="bottom")

        btn_cancel = tk.Button(bottom_frame, text="Annuler", width=14, font=font_bouton, command=self.concat_window.destroy)
        btn_cancel.pack(side="right", padx=5)

        btn_start = tk.Button(bottom_frame, text="Concaténer", width=14, font=font_bouton, command=self.concat_start)
        btn_start.pack(side="right", padx=5)

    def concat_add_files(self):
        files = filedialog.askopenfilenames(
            title="Sélectionnez des fichiers vidéo à ajouter",
            filetypes=[("Fichiers vidéo", "*.mp4 *.MP4")]
        )
        for f in files:
            if f not in self.listbox_files.get(0, tk.END):
                self.listbox_files.insert(tk.END, f)

    def concat_remove_selected(self):
        selected = list(self.listbox_files.curselection())
        if not selected:
            messagebox.showinfo("Information", "Aucune sélection à supprimer.")
            return
        for index in reversed(selected):
            self.listbox_files.delete(index)

    def concat_move_up(self):
        selected = list(self.listbox_files.curselection())
        if not selected:
            return
        for index in selected:
            if index == 0:
                continue
            text = self.listbox_files.get(index)
            self.listbox_files.delete(index)
            self.listbox_files.insert(index - 1, text)
            self.listbox_files.selection_set(index - 1)
            self.listbox_files.selection_clear(index)

    def concat_move_down(self):
        selected = list(self.listbox_files.curselection())
        if not selected:
            return
        size = self.listbox_files.size()
        for index in reversed(selected):
            if index == size - 1:
                continue
            text = self.listbox_files.get(index)
            self.listbox_files.delete(index)
            self.listbox_files.insert(index + 1, text)
            self.listbox_files.selection_set(index + 1)
            self.listbox_files.selection_clear(index)

    def concat_start(self):
        files = list(self.listbox_files.get(0, tk.END))
        if len(files) < 2:
            messagebox.showwarning("Attention", "Veuillez sélectionner au moins deux fichiers pour la concaténation.")
            return

        out_path = filedialog.asksaveasfilename(
            title="Enregistrer la vidéo concaténée sous",
            defaultextension=".mp4",
            filetypes=[("Fichiers MP4", "*.mp4")]
        )
        if not out_path:
            return

        self.concat_window.destroy()  # fermer la fenêtre modale

        def concat_thread():
            try:
                with tempfile.NamedTemporaryFile(mode="w", delete=False, suffix=".txt") as listfile:
                    for f in files:
                        escaped_path = f.replace("'", r"'\''")  # Échapper apostrophes simples
                        listfile.write("file '{}'\n".format(escaped_path))
                    listfile_path = listfile.name

                cmd = [
                    "ffmpeg",
                    "-y",
                    "-f", "concat",
                    "-safe", "0",
                    "-i", listfile_path,
                    "-c", "copy",
                    out_path
                ]
                subprocess.run(cmd, check=True)
                os.remove(listfile_path)
                messagebox.showinfo("Succès", f"Concaténation terminée avec succès :\n{out_path}")
            except subprocess.CalledProcessError as e:
                messagebox.showerror("Erreur ffmpeg", f"Erreur lors de la concaténation :\n{e}")
            except Exception as ex:
                messagebox.showerror("Erreur inattendue", f"{ex}")

        threading.Thread(target=concat_thread, daemon=True).start()


if __name__ == "__main__":
    root = tk.Tk()
    root.title("Video and GPX Utility")
    app = VideoGpxApp(root)
    root.mainloop()
