# Copyright (c) 2025, benilerouge.ddns.net
# Licensed under the MIT License.

import tkinter as tk
from tkinter import messagebox
import tempfile
import subprocess
import threading
import platform
import os

import pygame
pygame.mixer.init()

class Tooltip:
    def __init__(self, widget, text=''):
        self.widget = widget
        self.text = text
        self.tipwindow = None
        widget.bind("<Enter>", self.show_tip)
        widget.bind("<Leave>", self.hide_tip)
    def show_tip(self, event=None):
        if self.tipwindow or not self.text:
            return
        x = self.widget.winfo_rootx() + 30
        y = self.widget.winfo_rooty() + 30
        tw = tk.Toplevel(self.widget)
        self.tipwindow = tw
        tw.wm_overrideredirect(True)
        tw.wm_geometry(f"+{x}+{y}")
        label = tk.Label(
            tw, text=self.text, justify='left',
            background="#ffffe0", relief='solid', borderwidth=1,
            font=("tahoma", "9", "normal")
        )
        label.pack(ipadx=1)
    def hide_tip(self, event=None):
        if self.tipwindow:
            self.tipwindow.destroy()
        self.tipwindow = None

class AutoCloseInfo:
    def __init__(self, parent, message, duration=3000):
        self.win = tk.Toplevel(parent)
        self.win.overrideredirect(True)
        self.win.configure(bg="#ffffe0")
        label = tk.Label(self.win, text=message, bg="#ffffe0", font=("Arial", 13))
        label.pack(ipadx=20, ipady=10)
        self.win.update_idletasks()
        parent_x = parent.winfo_rootx()
        parent_y = parent.winfo_rooty()
        parent_w = parent.winfo_width()
        parent_h = parent.winfo_height()
        win_w = self.win.winfo_width()
        win_h = self.win.winfo_height()
        x = parent_x + (parent_w - win_w)//2
        y = parent_y + (parent_h - win_h)//2
        self.win.geometry(f"+{x}+{y}")
        self.win.after(duration, self.win.destroy)

def play_preview_window(video_path, start, end, mix_path=None, duration_label=None):
    """
    Lecture simple d'un extrait ou d'un mix/blend généré, sans fondu/transitions.
    """
    from edit_audio import temp_files
    from mutagen.mp3 import MP3

    preview_file = tempfile.NamedTemporaryFile(delete=False, suffix=".mp3")
    preview_file.close()
    temp_files.add(preview_file.name)

    try:
        if mix_path is None:
            preview_start = max(0, start - 3)
            preview_end = end + 3
            subprocess.run([
                "ffmpeg", "-y", "-i", video_path,
                "-ss", str(preview_start), "-to", str(preview_end),
                "-vn", "-acodec", "libmp3lame", preview_file.name
            ], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL, check=True)
        else:
            subprocess.run([
                "ffmpeg", "-y", "-i", mix_path,
                "-acodec", "libmp3lame", preview_file.name
            ], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL, check=True)

        # Récupérer la durée réelle de l'audio généré
        try:
            total_preview = MP3(preview_file.name).info.length
        except Exception:
            if mix_path is None:
                total_preview = preview_end - preview_start
            else:
                total_preview = end - start

        def _play(path, total_preview):
            try:
                if pygame.mixer.music.get_busy():
                    pygame.mixer.music.stop()
                pygame.mixer.music.load(path)
                pygame.mixer.music.play()

                if duration_label is not None:
                    interval = 0.2
                    elapsed = 0.0
                    while pygame.mixer.music.get_busy() and elapsed <= total_preview + 1:
                        text = f"{elapsed:.1f} / {total_preview:.1f} s"
                        duration_label.config(text=text)
                        duration_label.update_idletasks()
                        threading.Event().wait(interval)
                        elapsed += interval
                    duration_label.config(text="")
            except Exception as e:
                messagebox.showerror("Playback Error", str(e))

        threading.Thread(target=_play, args=(preview_file.name, total_preview), daemon=True).start()

    except Exception as e:
        if duration_label is not None:
            duration_label.config(text="")
        messagebox.showerror("Error", f"Preview generation failed: {e}")

class BlendDialog(tk.Toplevel):
    MANUAL_TOLERANCE = 0.1

    def __init__(self, master, start, end, video_path=None):
        from edit_audio import temp_files
        super().__init__(master)
        self.title("Paramètres de mixage")
        self.result = None
        self.start = start
        self.end = end
        self.video_path = video_path
        self.duration_target = end - start

        label = tk.Label(self, text=f"Sélection : {start:.1f} - {end:.1f} sec", font=("Arial", 11,))
        label.pack(pady=(10, 8))

        self.duration_target_label = tk.Label(
            self, text=f"Durée cible: {self.duration_target:.2f} s", font=("Arial", 9, "italic"))
        self.duration_target_label.pack()

        frame = tk.Frame(self)
        frame.pack(pady=2)

        tk.Label(frame, text="Volume du son existant :", font=("Arial", 10)).grid(row=0, column=0, padx=4, pady=2, sticky="e")
        self.vol_section = tk.Spinbox(frame, from_=0, to=100, width=5)
        self.vol_section.grid(row=0, column=1, padx=4)
        self.vol_section.delete(0, "end")
        self.vol_section.insert(0, "30")

        tk.Label(frame, text="Volume du son ajouté :", font=("Arial", 10)).grid(row=1, column=0, padx=4, pady=2, sticky="e")
        self.vol_prev = tk.Spinbox(frame, from_=0, to=100, width=5)
        self.vol_prev.grid(row=1, column=1, padx=4)
        self.vol_prev.delete(0, "end")
        self.vol_prev.insert(0, "100")

        tk.Label(self, text="(0 = silence, 100 = niveau original)").pack(pady=(4,0))

        self.blend_mode_var = tk.StringVar(value="before")
        frame_dir = tk.LabelFrame(self, text="Audio à utiliser pour le mixage", padx=6, pady=6)
        frame_dir.pack(pady=(4, 8))
        tk.Radiobutton(frame_dir, text="Avant la section sélectionnée", variable=self.blend_mode_var, value="before").pack(anchor="w")
        tk.Radiobutton(frame_dir, text="Après la section sélectionnée", variable=self.blend_mode_var, value="after").pack(anchor="w")
        tk.Radiobutton(frame_dir, text="Manuel", variable=self.blend_mode_var, value="manual").pack(anchor="w")

        self.manual_start_var = tk.StringVar()
        self.manual_end_var = tk.StringVar()
        frame_manual = tk.Frame(self)
        self.frame_manual = frame_manual
        tk.Label(frame_manual, text="Début (s):", font=("Arial", 9)).grid(row=0, column=0, padx=3)
        entry_start = tk.Entry(frame_manual, textvariable=self.manual_start_var, width=7)
        entry_start.grid(row=0, column=1)
        tk.Label(frame_manual, text="Fin (s):", font=("Arial", 9)).grid(row=1, column=0, padx=3)
        entry_end = tk.Entry(frame_manual, textvariable=self.manual_end_var, width=7)
        entry_end.grid(row=1, column=1)
        self.manual_duration_label = tk.Label(frame_manual, text="Durée spécifiée manuellement: N/A", font=("Arial", 9, "italic"))
        self.manual_duration_label.grid(row=2, column=0, columnspan=2)
        frame_manual.pack_forget()

        self.blend_mode_var.trace_add('write', self._on_blend_mode_change)
        self.manual_start_var.trace_add("write", self._update_manual_duration)
        self.manual_end_var.trace_add("write", self._update_manual_duration)

        self.use_micro_var = tk.BooleanVar(value=False)
        self.recorded_path = None
        mic_check = tk.Checkbutton(self, text="Enregistrement vocal (Commencez à parler dès que la notification s’affiche.\nImportant : Si nécessaire, ajuster aussi le niveau du micro sur le système d’exploitation.)", variable=self.use_micro_var)
        mic_check.pack(pady=(4,0))
        Tooltip(mic_check, "Enregistrer un commentaire vocal à mixer à la piste audio.")

        btn_frame = tk.Frame(self)
        btn_frame.pack(pady=10)

        record_btn = tk.Button(btn_frame, text="🎙 Parler", width=11, command=self._record_voice)
        record_btn.pack(side="left", padx=5)
        play_btn = tk.Button(btn_frame, text="🎧 Écouter", width=9, command=self._play_blend)
        play_btn.pack(side="left", padx=5)
        ok_btn = tk.Button(btn_frame, text="OK", width=9, command=self._on_ok)
        ok_btn.pack(side="left", padx=5)
        cancel_btn = tk.Button(btn_frame, text="Annuler", width=9, command=self._on_cancel)
        cancel_btn.pack(side="left", padx=5)
        self.ok_btn = ok_btn

        self.preview_duration_label = tk.Label(self, text="", font=("Arial", 10))
        self.preview_duration_label.pack()

        self._update_manual_duration()
        self.focus()
        self.grab_set()
        self.protocol("WM_DELETE_WINDOW", self._on_cancel)
        self.wait_window()

    def _on_blend_mode_change(self, *args):
        if self.blend_mode_var.get() == "manual":
            self.frame_manual.pack(pady=(4, 8))
            self._update_manual_duration()
        else:
            self.frame_manual.pack_forget()
            self.ok_btn.configure(state="normal")

    def _update_manual_duration(self, *args):
        if self.blend_mode_var.get() != "manual":
            self.ok_btn.configure(state="normal")
            self.manual_duration_label.config(text="Durée définie manuellement : N/A", fg="#666")
            return
        try:
            ms = float(self.manual_start_var.get())
            me = float(self.manual_end_var.get())
            duration = me - ms
            if duration <= 0:
                self.manual_duration_label.config(text="Durée définie manuellement : INVALIDE", fg="#c00")
                self.ok_btn.configure(state="disabled")
                return
            self.manual_duration_label.config(text=f"Durée définie manuellement : {duration:.2f} s", fg="#222")
            diff = abs(duration - self.duration_target)
            if diff > self.MANUAL_TOLERANCE:
                self.manual_duration_label.config(
                    text=f"Durée définie manuellement : {duration:.2f} s (diff {diff:.2f}s)", fg="#c60"
                )
                self.ok_btn.configure(state="disabled")
            else:
                self.ok_btn.configure(state="normal")
        except Exception:
            self.manual_duration_label.config(text="Durée définie manuellement : N/A", fg="#666")
            self.ok_btn.configure(state="disabled")

    def _record_voice(self):
        from edit_audio import temp_files
        duration = self.end - self.start
        save_path = tempfile.NamedTemporaryFile(delete=False, suffix=".wav").name
        temp_files.add(save_path)

        input_formats = {"Windows": "dshow", "Linux": "alsa", "Darwin": "avfoundation"}
        mic_devices = {"Windows": "audio=Microphone", "Linux": "default", "Darwin": ":0"}
        system = platform.system()
        fmt = input_formats.get(system, "dshow")
        mic = mic_devices.get(system, "default")

        try:
            AutoCloseInfo(self, "Enregistrement (Parlez...)", duration=1200)
            subprocess.run([
                "ffmpeg","-y", "-f", fmt, "-i", mic,
                "-t", f"{duration:.2f}", "-ac", "1", "-ar", "44100", save_path
            ], check=True)
            corrected_path = tempfile.NamedTemporaryFile(delete=False,suffix=".wav").name
            temp_files.add(corrected_path)
            subprocess.run([
                "ffmpeg", "-y", "-i", save_path,
                "-t", f"{duration:.2f}",
                "-af", "apad", corrected_path
            ], check=True)
            self.recorded_path = corrected_path
            messagebox.showinfo("Microphone", f"Enregistrement OK ({duration:.1f}s).")
        except Exception as e:
            messagebox.showerror("Erreur Microphone", f"Erreur : {e}")
            self.use_micro_var.set(False)

    def _on_ok(self):
        try:
            v_section = float(self.vol_section.get()) / 100.0
            v_prev = float(self.vol_prev.get()) / 100.0
            assert 0 <= v_section <= 1 and 0 <= v_prev <= 1
        except Exception:
            messagebox.showerror("Erreur de paramètre", "Saisir les volumes (0-100).")
            return
        blend_mode = self.blend_mode_var.get()
        manual_times = None
        if blend_mode == "manual":
            try:
                ms = float(self.manual_start_var.get())
                me = float(self.manual_end_var.get())
                duration = me - ms
                assert 0 <= ms < me
                if abs(duration - self.duration_target) > self.MANUAL_TOLERANCE:
                    messagebox.showerror("Erreur de saisie manuelle",
                        f"La durée définie manuellement ({duration:.2f}s) doit être égale à la durée cible ({self.duration_target:.2f}s).")
                    return
                manual_times = (ms, me)
            except Exception:
                messagebox.showerror("Erreur de saisie manuelle", "Saisissez des temps valides.")
                return
        self.result = (
            v_section, v_prev, blend_mode,
            self.use_micro_var.get(),
            self.recorded_path,
            manual_times
        )
        self.destroy()

    def _on_cancel(self):
        self.result = None
        self.destroy()

    def _play_blend(self):
        from edit_audio import temp_files

        s, e = self.start, self.end
        dur = e - s
        blend_mode = self.blend_mode_var.get()
        v_section = float(self.vol_section.get()) / 100.0
        v_prev = float(self.vol_prev.get()) / 100.0

        blend_a = tempfile.NamedTemporaryFile(delete=False, suffix=".mp3")
        blend_b = tempfile.NamedTemporaryFile(delete=False, suffix=".mp3")
        blend_mix = tempfile.NamedTemporaryFile(delete=False, suffix=".mp3")
        for f in [blend_a, blend_b, blend_mix]: f.close()
        temp_files.add(blend_a.name)
        temp_files.add(blend_b.name)
        temp_files.add(blend_mix.name)

        try:
            # Extraire l'audio original brut de la section
            subprocess.run([
                "ffmpeg", "-y", "-i", self.video_path, "-ss", str(s), "-to", str(e),
                "-acodec", "libmp3lame", blend_a.name
            ], check=True)

            if self.use_micro_var.get() and self.recorded_path:
                # Conversion de l'audio micro en stéréo 44.1kHz avec durée fixe
                subprocess.run([
                    "ffmpeg", "-y", "-i", self.recorded_path,
                    "-ac", "2", "-ar", "44100", "-t", str(dur),
                    blend_b.name
                ], check=True)
            else:
                if blend_mode == "manual":
                    ps = float(self.manual_start_var.get())
                    pe = float(self.manual_end_var.get())
                elif blend_mode == "before":
                    ps, pe = s - dur, s
                else:
                    ps, pe = e, e + dur

                # éviter de passer en négatif
                ps = max(0, ps)
                pe = max(ps, pe)

                subprocess.run([
                    "ffmpeg", "-y", "-i", self.video_path,
                    "-ss", str(ps), "-to", str(pe),
                    "-acodec", "libmp3lame", blend_b.name
                ], check=True)

            # Mixage avec volumes appliqués dans filter_complex pour effet réel
            filter_complex = (
                f"[0:a]volume={v_section}[a0];"
                f"[1:a]volume={v_prev}[a1];"
                "[a0][a1]amix=inputs=2:duration=first:dropout_transition=0"
            )
            subprocess.run([
                "ffmpeg", "-y",
                "-i", blend_a.name, "-i", blend_b.name,
                "-filter_complex", filter_complex,
                "-c:a", "libmp3lame", blend_mix.name
            ], check=True)

            # Lecture du mixage en preview
            play_preview_window(self.video_path, s, e, mix_path=blend_mix.name, duration_label=self.preview_duration_label)
        except Exception as e:
            self.preview_duration_label.config(text="")
            messagebox.showerror("Erreur audio", f"Erreur: {e}")

__all__ = ["Tooltip", "AutoCloseInfo", "BlendDialog", "play_preview_window"]
