# Copyright (c) 2025, benilerouge.ddns.net
# Licensed under the MIT License.

import tkinter as tk
from tkinter import ttk, filedialog, messagebox
import mpv
import threading
import time
import os
import subprocess
import tempfile

from edit_audio import Section, temp_files, apply_all_edits_func
from ui_widgets import Tooltip, AutoCloseInfo, BlendDialog
from video_GPX import VideoGpxApp


def nearest_keyframe(pos):
    K = 1.001
    n = int(round(pos / K))
    return round(n * K, 3)


class ProcessingDialog:
    def __init__(self, root, message="Traitement en cours... patientez"):
        self.top = tk.Toplevel(root)
        self.top.title("Traitement en cours")
        self.top.geometry("320x80")
        self.top.transient(root)
        self.top.grab_set()
        label = tk.Label(self.top, text=message, font=("Arial", 12))
        label.pack(pady=10)
        self.progress = ttk.Progressbar(self.top, mode='indeterminate', length=250)
        self.progress.pack(pady=5)
        self.progress.start(10)
        self.top.protocol("WM_DELETE_WINDOW", lambda: None)
        self.keep_temp_files = False  # False par défaut

    def close(self):
        self.progress.stop()
        self.top.grab_release()
        self.top.destroy()


def process_with_wait_dialog(root, func, *args, **kwargs):
    """
    Lance func(*args, **kwargs) dans un thread, affiche un dialog de traitement en attente si
    > 2s, ferme le dialog à la fin et renvoie la valeur de retour.
    """
    result = {}

    def target():
        try:
            result['value'] = func(*args, **kwargs)
        except Exception as e:
            result['exc'] = e

    t = threading.Thread(target=target)
    t.start()
    dialog = None
    start = time.time()
    while t.is_alive():
        root.update()
        if dialog is None and time.time() - start > 2:
            dialog = ProcessingDialog(root)
        time.sleep(0.05)
    if dialog:
        dialog.close()
    if 'exc' in result:
        messagebox.showerror("Error", str(result['exc']))
        raise result['exc']
    return result.get('value')


class MPVSectionSelector:
    def __init__(self, parent):
        self.root = parent
        self.bg_color = "#000000"
        self.text_color = "#FFFFFF"
        self.player = mpv.MPV(
            input_default_bindings=False,
            input_vo_keyboard=False,
            vo='gpu',
            keep_open='yes'
        )
        self.player.loop = 'no'
        self.video_loaded = False
        self.sections = []
        self.section_start = None
        self.duration = 0
        self.video_path = None
        self.initial_video_path = None
        self.is_paused = True
        self.current_speed = 1.0

        script_dir = os.path.dirname(os.path.abspath(__file__))
        self.temp_dir = os.path.join(script_dir, "temp")
        os.makedirs(self.temp_dir, exist_ok=True)

        self.tmp_file = None

        # Liste pour stocker les positions des découpes libres F/D (hors keyframes)
        self.fd_cut_points = []

        self.root.winfo_toplevel().protocol("WM_DELETE_WINDOW", self.on_close)

        self.frame = tk.Frame(self.root, bg=self.bg_color)
        self.frame.pack(fill="both", expand=True)

        self.init_buttons()

        self.label = tk.Label(
            self.frame,
            text="Position: 00:00 (0.000s) / 00:00 (0.000s) | Speed: 1.000x",
            bg=self.bg_color,
            fg=self.text_color
        )
        self.label.grid(row=4, column=0, columnspan=9, pady=(8, 0), sticky="we")

        self.scale = tk.Scale(
            self.frame,
            from_=0,
            to=100,
            orient='horizontal',
            length=600,
            showvalue=0,
            bg=self.bg_color,
            troughcolor="#e0e0e0",
            highlightthickness=0,
            command=self.on_scale_move
        )
        self.scale.grid(row=5, column=0, columnspan=9, pady=(6, 0), sticky="we")
        self.scale_drag = False
        self.scale.bind("<Button-1>", self.on_scale_press)
        self.scale.bind("<ButtonRelease-1>", self.on_scale_release)

        @self.player.property_observer('time-pos')
        def time_observer(_name, value):
            if value is not None:
                cur_pos = value or 0
                cur_pos_minsec = self.format_time_min_sec(cur_pos)
                dur_minsec = self.format_time_min_sec(self.duration)
                self.label.config(
                    text=(
                        f"Position: {cur_pos_minsec} ({cur_pos:.3f}s) / "
                        f"{dur_minsec} ({self.duration:.3f}s) | "
                        f"Speed: {self.current_speed:.3f}x"
                    )
                )
                if self.duration > 0 and not self.scale_drag:
                    self.scale.set(value * 100 / self.duration)

        @self.player.property_observer('duration')
        def duration_observer(_name, value):
            if value is not None:
                self.duration = value
                cur_pos = self.player.time_pos or 0
                cur_pos_minsec = self.format_time_min_sec(cur_pos)
                dur_minsec = self.format_time_min_sec(self.duration)
                self.label.config(
                    text=(
                        f"Position: {cur_pos_minsec} ({cur_pos:.3f}s) / "
                        f"{dur_minsec} ({self.duration:.3f}s) | "
                        f"Speed: {self.current_speed:.3f}x"
                    )
                )
                self.scale.config(to=100)

        @self.player.property_observer('pause')
        def pause_observer(_name, value):
            self.is_paused = value
            self.update_play_pause_button()

        @self.player.property_observer('speed')
        def speed_observer(_name, value):
            if value is not None:
                self.current_speed = float(value)
                cur_pos = self.player.time_pos or 0
                cur_pos_minsec = self.format_time_min_sec(cur_pos)
                dur_minsec = self.format_time_min_sec(self.duration)
                self.label.config(
                    text=(
                        f"Position: {cur_pos_minsec} ({cur_pos:.3f}s) / "
                        f"{dur_minsec} ({self.duration:.3f}s) | "
                        f"Speed: {self.current_speed:.3f}x"
                    )
                )

        self.root.bind_all("<space>", lambda e: self.toggle_play_pause())
        self.root.bind_all("<Left>", lambda e: self.rewind_1s())
        self.root.bind_all("<Right>", lambda e: self.forward_1s())
        self.root.bind_all("<Up>", lambda e: self.frame_back())
        self.root.bind_all("<Down>", lambda e: self.frame_forward())
        self.root.bind_all("<i>", lambda e: self.mark_section_start())
        self.root.bind_all("<I>", lambda e: self.mark_section_start())
        self.root.bind_all("<o>", lambda e: self.mark_section_end())
        self.root.bind_all("<O>", lambda e: self.mark_section_end())
        self.root.bind_all("<Control-Shift-F>", lambda e: self._run_with_processing_dialog(self.cut_from_start_to_current_tempfile))
        self.root.bind_all("<Control-Shift-D>", lambda e: self._run_with_processing_dialog(self.cut_from_current_to_end_concat))
        self.root.bind_all("<Shift-Control-Left>", lambda e: self._run_with_processing_dialog(self.cut_remove_start_and_open))
        self.root.bind_all("<Shift-Control-Right>", lambda e: self._run_with_processing_dialog(self.cut_remove_end_and_open))

        self.root.focus_set()

    def _run_with_processing_dialog(self, func, *args, **kwargs):
        """Lance une fonction avec affichage du dialog 'Traitement en cours' s’il dure plus de 2 secondes."""
        process_with_wait_dialog(self.root, func, *args, **kwargs)

    def on_close(self):
        try:
            if self.player:
                self.player.terminate()
        except Exception:
            pass
        self.root.winfo_toplevel().destroy()

    def format_time_min_sec(self, seconds):
        if seconds is None or seconds < 0:
            return "00:00"
        m = int(seconds) // 60
        s = int(seconds) % 60
        return f"{m:02d}:{s:02d}"

    def _block_default_keys_in_children(self, parent):
        keys = ["<space>", "<Left>", "<Right>", "<Up>", "<Down>"]
        for child in parent.winfo_children():
            if isinstance(child, (tk.Button, ttk.Scale, tk.Scale)):
                for key in keys:
                    child.bind(key, lambda e: "break")
            if isinstance(child, (tk.Frame, ttk.Frame)):
                self._block_default_keys_in_children(child)

    def init_buttons(self):
        row1_frame = tk.Frame(self.frame, bg=self.bg_color)
        row1_frame.grid(row=0, column=0, columnspan=9, pady=2, sticky="we")
        row1 = [
            ("📂", self.open_file, "Ouvrir un fichier vidéo"),
            ("⏪", self.frame_back, "Image clé précédente (↑)"),
            ("<<", self.rewind_1s, "Recul de 1s (←)"),
            ("⏯", self.toggle_play_pause, "Lecture/Pause (space)"),
            (">>", self.forward_1s, "Avancer de 1s (→)"),
            ("⏩", self.frame_forward, "Image clé suivante (↓)"),
            ("1x", lambda: self.set_speed(1.0), "Vitesse 1x"),
            ("2x", lambda: self.set_speed(2.0), "Vitesse 2x"),
            ("3x", lambda: self.set_speed(3.0), "Vitesse 3x"),
            ("4x", lambda: self.set_speed(4.0), "Vitesse 4x"),
        ]
        for txt, cmd, tip in row1:
            btn = tk.Button(
                row1_frame, text=txt, command=cmd,
                width=4, bg="#fdcc24",
                font=("Arial", 11),
                relief="raised"
            )
            btn.pack(side="left", padx=5, pady=2, expand=True, fill="x")
            Tooltip(btn, tip)
            if txt == "⏯":
                self.play_pause_btn = btn

        row2_frame = tk.Frame(self.frame, bg=self.bg_color)
        row2_frame.grid(row=1, column=0, columnspan=9, pady=2, sticky="we")
        row2 = [
            ("[", self.mark_section_start, "Début de section (I)"),
            ("]", self.mark_section_end, "Fin de section (O)"),
            ("Sections", self.show_sections, "Sections sélectionnées"),
            ("Annuler", self.undo_last_section, "Annuler dernière sélection"),
            ("Annuler tout", self.undo_all_sections, "Annuler toutes les sélections"),
            ("Exécuter", self.apply_all_edits, "Créer le fichier vidéo"),
        ]
        for i, (txt, cmd, tip) in enumerate(row2):
            btn = tk.Button(
                row2_frame, text=txt, command=cmd,
                width=10, bg="#fdcc24",
                fg="#222",
                font=("Arial", 10)
            )
            btn.pack(side="left", padx=6, pady=2, expand=True, fill="x")
            Tooltip(btn, tip)

    def open_file(self):
        path = filedialog.askopenfilename(filetypes=[
            ("Video files", "*.mp4 *.MP4")
        ])
        if path:
            self.video_path = path
            self.initial_video_path = path
            menu_x = self.root.winfo_rootx()
            menu_y = self.root.winfo_rooty()
            menu_width = self.root.winfo_width()
            video_height = 500
            geometry = f"{menu_width}x{video_height}+{menu_x}+{menu_y + self.root.winfo_height()}"
            self.player.geometry = geometry

            threading.Thread(
                target=lambda: self.player.command('loadfile', path, 'replace', 'pause=yes'),
                daemon=True
            ).start()

            self.video_loaded = True
            self.sections.clear()
            self.section_start = None
            self.is_paused = True
            self.update_play_pause_button()
            init_minsec = self.format_time_min_sec(0)
            self.label.config(
                text=f"Position : {init_minsec} (0.000s) / {init_minsec} (0.000s) | Vitesse : {self.current_speed:.1f}x"
            )
            self.root.after(200, self.root.focus_force)


    def frame_back(self):
        if self.video_loaded:
            self.player.command('frame-back-step')

    def frame_forward(self):
        if self.video_loaded:
            self.player.command('frame-step')

    def rewind_1s(self):
        if self.video_loaded:
            self.player.seek(-1, reference='relative')

    def forward_1s(self):
        if self.video_loaded:
            self.player.seek(1, reference='relative')

    def toggle_play_pause(self):
        if self.video_loaded:
            self.player.pause = not self.player.pause

    def mark_section_start(self):
        if self.video_loaded and self.player.time_pos is not None:
            self.section_start = self.player.time_pos
            AutoCloseInfo(self.root, f"Début de section : {self.section_start:.3f} s", duration=3000)
        else:
            AutoCloseInfo(self.root, "No video loaded or unknown position.", duration=3000)

    def mark_section_end(self):
        if self.video_loaded and self.player.time_pos is not None:
            if self.section_start is not None and self.player.time_pos > self.section_start:
                start = self.section_start
                end = self.player.time_pos
                self.section_start = None

                blend_win = BlendDialog(self.root, start, end, self.video_path)
                if blend_win.result is not None:
                    v_section, v_prev, blend_mode, use_micro, mic_path, manual_times = blend_win.result
                    section = Section(start, end, "blend", v_section, v_prev, blend_mode)
                    section.use_micro = use_micro
                    section.mic_path = mic_path
                    if blend_mode == "manual" and manual_times is not None:
                        section.manual_times = manual_times
                    self.sections.append(section)
                    msg = (f"Blend section added: {start:.3f}-{end:.3f}s "
                           f"({int(v_section*100)}/{int(v_prev*100)} %, {blend_mode}, mic={use_micro}")
                    if blend_mode == "manual" and manual_times:
                        msg += f", {manual_times[0]:.3f}-{manual_times[1]:.3f}s"
                    msg += ")"
                    AutoCloseInfo(self.root, msg, duration=2500)
            else:
                AutoCloseInfo(self.root, "Mark section start (I) and then end (O).", duration=3000)
        else:
            AutoCloseInfo(self.root, "No video loaded, or unknown position.", duration=3000)


    def cut_from_start_to_current_tempfile(self):
        """Découpe depuis la position courante jusqu'à la fin, crée un fichier temporaire."""
        if not (self.video_loaded and self.video_path and self.player.time_pos is not None):
            messagebox.showwarning("Erreur", "Vidéo non chargée ou position inconnue.")
            return
        pos = self.player.time_pos

        # Ajout position dans liste fd_cut_points pour vérification plus tard
        if pos is not None:
            self.fd_cut_points.append(pos)

        tmpdir = self.temp_dir
        base, ext = os.path.splitext(os.path.basename(self.video_path))
        temp_name = f"temp_cut_from_start_{base}{ext}"
        temp_path = os.path.join(tmpdir, temp_name)

        cmd = [
            "ffmpeg",
            "-y",
            "-ss", f"{pos:.3f}",
            "-i", self.video_path,
            "-c", "copy",
            "-avoid_negative_ts", "make_zero",
            temp_path
        ]
        try:
            subprocess.run(cmd, check=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            self.tmp_file = temp_path
            messagebox.showinfo("Temp file créé",
                                f"La partie depuis {pos:.3f}s est enregistrée dans un fichier temporaire.\n"
                                f"Utilise Ctrl+Shift+D pour concaténer avec la partie avant.")
        except Exception as e:
            messagebox.showerror("Erreur",
                                 f"Erreur lors de la création du fichier temporaire :\n{e}")

    def cut_from_current_to_end_concat(self):
        """Concatène la partie de début jusqu’à la position courante avec le fichier temporaire."""
        if not (self.video_loaded and self.video_path and self.player.time_pos is not None):
            messagebox.showwarning("Erreur", "Vidéo non chargée ou position inconnue.")
            return
        if not self.tmp_file or not os.path.exists(self.tmp_file):
            messagebox.showwarning("Pas de fichier temporaire",
                                   "Le fichier temporaire n'existe pas.\nUtilise d'abord Ctrl+Shift-F.")
            return

        pos = self.player.time_pos
        tmpdir = self.temp_dir
        base, ext = os.path.splitext(os.path.basename(self.video_path))
        part_head = os.path.join(tmpdir, f"temp_cut_to_{base}{ext}")
        output_concat = os.path.join(tmpdir, f"concat_result_{base}{ext}")

        try:
            cmd_head = [
                "ffmpeg",
                "-y",
                "-i", self.video_path,
                "-to", f"{pos:.3f}",
                "-c", "copy",
                "-avoid_negative_ts", "make_zero",
                part_head
            ]
            subprocess.run(cmd_head, check=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)

            concat_txt = os.path.join(tmpdir, "concat_list.txt")
            with open(concat_txt, "w") as f:
                f.write(f"file '{part_head}'\n")
                f.write(f"file '{self.tmp_file}'\n")

            cmd_concat = [
                "ffmpeg",
                "-y",
                "-f", "concat",
                "-safe", "0",
                "-i", concat_txt,
                "-c", "copy",
                output_concat
            ]
            subprocess.run(cmd_concat, check=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)

            self.video_path = output_concat
            self.player.command('loadfile', output_concat, 'replace', 'pause=yes')
            messagebox.showinfo("Concaténation terminée",
                                f"Fichier final créé et chargé :\n{output_concat}")

            for f in [part_head, self.tmp_file, concat_txt]:
                if os.path.exists(f):
                    os.remove(f)
            self.tmp_file = None

            # Vérification intégrité aux points F/D
            success = self.verify_fd_cut_points_integrity()
            if not success:
                messagebox.showerror("Attention", "Des erreurs ont été détectées autour des points de découpe F/D.")

        except Exception as e:
            messagebox.showerror("Erreur", f"Erreur lors de la concaténation :\n{e}")

    def verify_fd_cut_points_integrity(self, buffer=1.5, segment_duration=3.0):
        """Vérifie la qualité de la vidéo uniquement autour des points de découpe F/D."""
        if not self.fd_cut_points:
            messagebox.showinfo("Vérification", "Aucun point de découpe F/D enregistré.")
            return True

        video_path = self.video_path
        for pos in self.fd_cut_points:
            start = max(pos - buffer, 0)
            cmd = [
                "ffmpeg", "-v", "error",
                "-ss", f"{start:.3f}",
                "-i", video_path,
                "-t", f"{segment_duration:.3f}",
                "-f", "null",
                "-"
            ]
            result = subprocess.run(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True)
            if result.stderr.strip():
                messagebox.showwarning(
                    "Erreur détectée",
                    f"Erreur détectée près du point de découpe {pos:.3f} s :\n{result.stderr}"
                )
                return False
        messagebox.showinfo("Vérification", "Aucun problème détecté aux points de découpe F/D.")
        return True

    def show_sections(self):
        if not self.sections:
            messagebox.showinfo("Sections", "No section selected.")
            return
        txt = ""
        for i, s in enumerate(self.sections):
            txt += f"{i + 1}: {s}\n"
        messagebox.showinfo("Sections", txt)

    def undo_last_section(self):
        if self.sections:
            self.sections.pop()
            messagebox.showinfo("Undo Last", "Last selection deleted.")
        else:
            messagebox.showinfo("Undo Last", "No selection to cancel.")

    def undo_all_sections(self):
        if self.sections:
            self.sections.clear()
            messagebox.showinfo("Undo All", "All selections deleted.")
        else:
            messagebox.showinfo("Undo All", "No selections to delete.")

    def set_speed(self, speed):
        if self.video_loaded:
            self.player.speed = speed
            self.current_speed = speed
            cur_pos = self.player.time_pos or 0
            cur_pos_minsec = self.format_time_min_sec(cur_pos)
            dur_minsec = self.format_time_min_sec(self.duration)
            self.label.config(
                text=(
                    f"Position: {cur_pos_minsec} ({cur_pos:.3f}s) / "
                    f"{dur_minsec} ({self.duration:.3f}s) | "
                    f"Speed: {speed:.3f}x"
                )
            )

    def on_scale_move(self, value):
        if self.video_loaded and self.duration > 0 and self.scale_drag:
            pos = float(value) * self.duration / 100
            self.player.seek(pos, reference='absolute')

    def on_scale_press(self, event):
        self.scale_drag = True

    def on_scale_release(self, event):
        self.scale_drag = False
        self.on_scale_move(self.scale.get())

    def update_play_pause_button(self):
        if hasattr(self, "play_pause_btn"):
            if self.is_paused:
                self.play_pause_btn.config(text="▶")
            else:
                self.play_pause_btn.config(text="⏸")

    def apply_all_edits(self):
        if not self.sections:
            messagebox.showwarning("Error", "No section selected.")
            return
        if not self.video_path:
            messagebox.showwarning("Error", "Unknown video path.")
            return
        try:
            final_name = process_with_wait_dialog(
                self.root,
                apply_all_edits_func,
                self.video_path,
                self.duration,
                self.sections,
                self.temp_dir
            )
            self.video_path = final_name
            self.player.command('loadfile', self.video_path, 'replace', 'pause=yes')
            self.sections.clear()
            self.section_start = None
            messagebox.showinfo("Success", f"Processed.\nNew file: {final_name}")
        except Exception as e:
            messagebox.showerror("Error", f"Process error: {e}")

    def clean_files(self):
        if not self.initial_video_path or not self.video_path:
            messagebox.showwarning("CLEAN", "No video to protect.")
            return

        temp_files.clean(keep=[self.initial_video_path, self.video_path])
        msg = "Temporary files cleaned.\n"
        msg += f"Files kept:\n- {os.path.basename(self.initial_video_path)}\n- {os.path.basename(self.video_path)}"
        filename_base = os.path.splitext(self.initial_video_path)[0]
        for suffix in ["_modified.mp4", "_cut.mp4"]:
            candidate = filename_base + suffix
            if os.path.exists(candidate) and candidate != self.video_path:
                try:
                    os.remove(candidate)
                except Exception:
                    pass
        messagebox.showinfo("CLEAN", msg)

    def cut_remove_start_and_open(self):
        """Enlève le début jusqu’à la position courante (garde la fin)."""
        if not (self.video_loaded and self.video_path and self.player.time_pos is not None):
            messagebox.showwarning("Erreur", "Vidéo non chargée ou position inconnue.")
            return
        pos = self.player.time_pos
        if pos <= 0:
            messagebox.showinfo("Info", "Déjà au début de la vidéo, rien à enlever.")
            return
        tmpdir = self.temp_dir
        base, ext = os.path.splitext(os.path.basename(self.video_path))
        temp_name = f"temp_removed_start_{base}{ext}"
        temp_path = os.path.join(tmpdir, temp_name)

        try:
            cmd = [
                "ffmpeg",
                "-y",
                "-ss", f"{pos:.3f}",
                "-i", self.video_path,
                "-c", "copy",
                "-avoid_negative_ts", "make_zero",
                temp_path
            ]
            subprocess.run(cmd, check=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            self.video_path = temp_path
            self.player.command('loadfile', temp_path, 'replace', 'pause=yes')
            messagebox.showinfo("Succès", f"Début enlevé, fichier chargé:\n{temp_path}")
        except Exception as e:
            messagebox.showerror("Erreur", f"Erreur lors de la création du fichier temporaire :\n{e}")

    def cut_remove_end_and_open(self):
        """Enlève la fin à partir de la position courante (garde le début)."""
        if not (self.video_loaded and self.video_path and self.player.time_pos is not None):
            messagebox.showwarning("Erreur", "Vidéo non chargée ou position inconnue.")
            return
        pos = self.player.time_pos
        if self.duration - pos <= 0:
            messagebox.showinfo("Info", "Déjà à la fin de la vidéo, rien à enlever.")
            return
        tmpdir = self.temp_dir
        base, ext = os.path.splitext(os.path.basename(self.video_path))
        temp_name = f"temp_removed_end_{base}{ext}"
        temp_path = os.path.join(tmpdir, temp_name)

        try:
            cmd = [
                "ffmpeg",
                "-y",
                "-i", self.video_path,
                "-to", f"{pos:.3f}",
                "-c", "copy",
                "-avoid_negative_ts", "make_zero",
                temp_path
            ]
            subprocess.run(cmd, check=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            self.video_path = temp_path
            self.player.command('loadfile', temp_path, 'replace', 'pause=yes')
            messagebox.showinfo("Succès", f"Fin enlevée, fichier chargé:\n{temp_path}")
        except Exception as e:
            messagebox.showerror("Erreur", f"Erreur lors de la création du fichier temporaire :\n{e}")

    def cut_and_save_sections(self):
        try:
            self.cut_sections()
            if self.video_path:
                messagebox.showinfo("Succès", f"La découpe a été sauvegardée dans :\n{self.video_path}")
        except Exception as e:
            messagebox.showerror("Erreur", f"Erreur lors de la découpe : {e}")



if __name__ == "__main__":
    root = tk.Tk()
    root.title("Kinodioraptor")
    root.configure(bg="#000000")
    screen_width = root.winfo_screenwidth()
    window_width = 760
    window_height = 200
    x = (screen_width // 2) - (window_width // 2)
    y = 0
    root.geometry(f"{window_width}x{window_height}+{x}+{y}")

    notebook = ttk.Notebook(root)
    notebook.pack(fill="both", expand=True)

    tab_audio = tk.Frame(notebook, bg="#000000")
    tab_video = tk.Frame(notebook)

    notebook.add(tab_audio, text="Lecteur")
    notebook.add(tab_video, text="Outils")

    # --- Ajout de l'onglet Mode d’emploi ---

    tab_shortcuts = tk.Frame(notebook, bg="#000000")
    notebook.add(tab_shortcuts, text="Mode d’emploi")

    shortcuts_text = """
PETIT MODE D’EMPLOI

Le programme permet trois types d’opérations : édition audio, découpage vidéo sans réencodage et édition gpx.

Onglet [Lecteur]

👉 Édition audio
L’édition audio se fait dans l’onglet [Lecteur]. Les boutons et leurs raccourcis-clavier permettent de sélectionner une section, d’ajuster son niveau sonore et de lui superposer le son d’une autre section de la vidéo. Il est également possible d’ajouter des commentaires avec le micro de l’ordinateur.

👉 Découpage vidéo
Le découpage vidéo, sans réencodage, se fait lui aussi dans l’onglet [Lecteur], uniquement avec des raccourcis-clavier. Il s’agit principalement de couper les arrêts aux feux rouges, les erreurs de parcours, etc.

Ctrl+Shift+F (Fin) : Crée la partie 2 à conserver (après feu rouge, etc.)
Ctrl+Shift+D (Début) : Crée la partie 1 à conserver (avant feu rouge, etc.)
→ La sélection se fait dans cet ordre, pour assurer la précision de la transition. On choisit l’image clé où commence la partie 2, puis, pour la partie 1, l’image la plus adéquate entre deux images clés. Il n’est pas nécessaire de choisir une image clé pour la fin de la partie 1, et c’est ce qui permet une transition précise entre celle-ci et l’image clé où commence la partie 2.
Shift+Ctrl+Flèche Gauche : Enlève tout le début de la vidéo, jusqu’à la position de lecture
Shift+Ctrl+Flèche Droite : Enlève tout de la position de lecture à la fin de la vidéo

!!! IMPORTANT !!!  IL NE FAUT JAMAIS EFFECTUER DES OPÉRATIONS VIDÉO ET AUDIO SUR UN MÊME FICHIER OUVERT DANS LE LECTEUR (AUTREMENT, L’UNE OU L’AUTRE DE CES OPÉRATIONS SERA IGNORÉE). LA PROCÉDURE LA PLUS LOGIQUE EST DE D’ABORD DÉCOUPER LA VIDÉO, PUIS, DANS LE NOUVEAU FICHIER OBTENU, DE PASSER AUX OPÉRATIONS AUDIO.

------

Onglet [Outils]

[Concaténer des vidéos] : Pour concaténer des fichiers vidéos enregistrés séparément par la caméra

[Lisser un GPX] : Pour adoucir la pente d’un fichier gpx et éliminer les incohérences (changements brusques d’altitude invraisemblables)

[Éditeur GPX] : Pour ouvrir le tracé gpx sur une carte, manipuler les points et resynchroniser le tracé après sa modification (feux rouges, etc.)

Couper le début et la fin :
Définir d’abord le point de début de la balade en cliquant sur un point (clic de souris ou touche Espace). Cliquer ensuite sur le bouton [Début ]. La carte se recharge avec le point de début défini sur le temps 00:00:00. Il faut attendre que la carte se soit rechargée, puis procéder de la même façon pour définir le point de fin de la balade en cliquant sur [Fin].

Couper des parties inutiles et resynchroniser le tracé :
Cliquer sur le premier point (souris ou Espace). Taper sur la touche Entrée pour activer le carré vert. Se déplacer jusqu’au deuxième point (fin de section à couper) en cliquant dessus avec la souris en en sélectionnant avec la touche Espace. Taper sur Entrée pour activer le carré vert. Si c’est la seule partie à couper, taper sur la lettre « o » (pour output). Cela crée le nouveau fichier. S’il y a d’autres parties à couper, continuer de mettre des carrés verts aux endroits désirés, puis, à la fin, cliquer sur « o » pour créer le fichier.

→ La resynchronisation s’effectue automatiquement. Le programme est conçu pour mettre un intervalle de 1 seconde entre le point de début de coupure et le point choisi pour la fin de la coupure. Ces deux points NE FONT PAS PARTIE de la partie coupée.


"""

    frame_text = tk.Frame(tab_shortcuts, bg="#000000")
    frame_text.pack(fill="both", expand=True, padx=10, pady=10)

    txt = tk.Text(frame_text, bg="#FFFFFF", fg="#000000", font=("Arial", 12))
    txt.insert("1.0", shortcuts_text)
    txt.config(state="disabled", wrap="word", relief="flat")
    txt.pack(side="left", fill="both", expand=True)

    scrollbar = tk.Scrollbar(frame_text, orient="vertical", command=txt.yview)
    scrollbar.pack(side="right", fill="y")

    txt.config(yscrollcommand=scrollbar.set)

    app_audio = MPVSectionSelector(tab_audio)
    app_video = VideoGpxApp(tab_video, get_video_path=lambda: app_audio.video_path)

    def on_tab_change(event):
        root.focus_set()
    notebook.bind("<<NotebookTabChanged>>", on_tab_change)

    root.mainloop()
